//MIDlet for writing go-kifus (i.e. game protocols)
//$Id: mgo_x.java,v 1.69 2007/05/10 15:42:18 der_hannes Exp $
//
//
// Copyright 2005, 2006, 2007 Hannes Michalek, parts added by matejcik
// 
// Distributed under the terms of the GNU General Public License (GPL):
// **************************************************************
// * This file is part of mgo.
// * 
// * mgo is free software; you can redistribute it and/or modify
// * it under the terms of the GNU General Public License as published by
// * the Free Software Foundation; either version 2 of the License, or
// * (at your option) any later version.
// *
// * mgo is distributed in the hope that it will be useful,
// * but WITHOUT ANY WARRANTY; without even the implied warranty of
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// * GNU General Public License for more details.
// *
// * You should have received a copy of the GNU General Public License
// * along with this program; if not, write to the Free Software
// * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
// **************************************************************
//
// =====================================
//	The Bouncycastle Crypto API used in mgo has been released under 
//	its own license, cf. http://www.bouncycastle.org/licence.html. 
//  Here's a copy (dated 2008/03/30):
//
//  Copyright (c) 2000 - 2006 The Legion Of The Bouncy Castle (http://www.bouncycastle.org)
//
//  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
//
//  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
//
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
//	====================================
//
//
// For further information on mgo and in case you want to 
// contact the developers, see http://www.mobilego.org.
//
// 	----------------------
// 	historical quotation of jabber-chat on 01.05.2005:
//	[18:17:51] <mathi> schwarz 
//	[18:17:54] <mathi> 1
//	[18:17:58] <mathi> weiss 2
//	[18:18:03] <mathi> keiner: 0
//	------------------------
// addendum: sequence with value "-1": passed move

import java.util.*;
import java.io.*;

import javax.microedition.io.Connection;
import javax.microedition.io.Connector;
import javax.microedition.io.HttpConnection;
import javax.microedition.io.StreamConnection;
import javax.microedition.io.file.FileConnection;
import javax.microedition.io.file.FileSystemRegistry;
import javax.microedition.lcdui.*;
import javax.microedition.midlet.MIDlet;
import javax.microedition.rms.RecordStore;
import javax.microedition.rms.RecordStoreException;
import javax.microedition.*;

import org.bouncycastle.util.encoders.Base64;


public class mgo_x extends MIDlet implements CommandListener {
	// the MIDlet itself
	
	private static String cvs_Id = "$Id: mgo_x.java,v 1.69 2007/05/10 15:42:18 der_hannes Exp $";
	
	//fileConnections possible? If not, this is "null"
	String fileConnectionVersion=System.getProperty("microedition.io.file.FileConnection.version");
	
	// variable initializations
	public static Display display;
	// I dearly hope this "static" doesn't mess something up
	
	GoBrett brett = null;

	private Command cmd_select = new Command ("Select", Command.ITEM, 1);
	private static Command cmd_ok = new Command("OK", Command.OK, 1);
	private Command cmd_cancel = new Command ("Back", Command.CANCEL, 2);
	private static Command cmd_back = new Command("BACK", Command.OK, 2);
		// commands need to go before ...er ... something where they're used in constructor
	
	private static String EGOBAN_USER = "egobanUserEh5uYaux";
	private static String EGOBAN_PASSWD = "egobanPasswdEh5uYaux";
	
	private static String SHOW_HELP_AT_STARTUP = "showHelpAtStartupEh5uYaux";
	
	private List mainMenu;
	private List ingameMenu;
	private List sizeSelection;
	private SaveSlotsList saveSlots = new SaveSlotsList();
	private SysInfoScreen sysinfo = new SysInfoScreen();
	private HelpScreen help = new HelpScreen();
	private ScoreScreen score = new ScoreScreen();
	private CaptsScreen capts = new CaptsScreen();
	private LogoScreen logoScreen = new LogoScreen(); 

	// goban needs to go after score
	private DrawingGoban goban = new DrawingGoban();
	
	private Preferences globalPreferences = new Preferences();
	
	private String[] menuItems = new String[] {
			"New kifu",
			"Load",
			"System info",
			"Settings",
			"Exit"
	};
	
	private String[] ingameMenuItems = new String[] {
			"Save game",
			"Set Review mode",
			"Set Play mode",
			"Count points",
			"Settings",
			"Main menu"
	};
	
	private int[] boardSizes = {19, 15, 13, 11, 9};
	private int[][] boardHoshis = {
			{4, 10, 16},
			{4, 8, 12},
			{4, 7, 10},
			{4, 8},
			{3, 7}
	};
	
	public mgo_x ()
	{
		display = Display.getDisplay(this);
		
		mainMenu = new List ("Main Menu", List.IMPLICIT, menuItems, null);
		mainMenu.setCommandListener (this);
		mainMenu.addCommand (cmd_select);
		
		sizeSelection = new List ("Select Board Size", List.IMPLICIT);
		for (int i = 0; i < boardSizes.length; i++) {
			String size = Integer.toString(boardSizes[i]);
			sizeSelection.append(size+"x"+size, null);
		}
		sizeSelection.setCommandListener (this);
		sizeSelection.addCommand (cmd_cancel);
		sizeSelection.addCommand (cmd_select);
		
		ingameMenu = new List ("Game Menu", List.IMPLICIT, ingameMenuItems, null);
		ingameMenu.setCommandListener (this);
		ingameMenu.addCommand(cmd_cancel);
		ingameMenu.addCommand(cmd_select);
		
		sysinfo.setCommandListener(this);
		sysinfo.addCommand(cmd_ok);		
		
	}
	
	public void commandAction (Command c, Displayable d)
	{
		if (d == mainMenu) {
			if (c == List.SELECT_COMMAND || c == cmd_select) {
				switch (mainMenu.getSelectedIndex()) {
					case 0: // new game
						display.setCurrent(sizeSelection);
						return;
					case 1: // load game
						SaveSlotsList saveSlots=new SaveSlotsList(); //clear List
						// XXX i'm not entirely convinced this is the correct way to do it
						saveSlots.invoke(d,SaveSlotsList.MODE_LOAD);
						return;
					case 2: // system info
						display.setCurrent(sysinfo);
						return;
					case 3: // change settings
						globalPreferences.initChangeSettings(d,null);
						return;
					case 4: // exit
						destroyApp(false);
						notifyDestroyed();
						return;
				}
			}
		} else if (d == sizeSelection) {
			if (c == List.SELECT_COMMAND || c == cmd_select) {
				/* initiate new game */
				goban.setMode (DrawingGoban.MODE_PLAY);
			    goban.setHoshiIndex(sizeSelection.getSelectedIndex());
			    brett = new GoBrett(boardSizes[sizeSelection.getSelectedIndex()]);
			    
				display.setCurrent(goban);
				return;
			} else if (c == cmd_cancel) {
				display.setCurrent(mainMenu);
				return;
			}
		} else if (d == ingameMenu) {
			if (c == List.SELECT_COMMAND || c == cmd_select) {
				switch (ingameMenu.getSelectedIndex()) {
					case 0: // save
						SaveSlotsList saveSlots=new SaveSlotsList(); //clear List
						// XXX see above
						saveSlots.invoke(d,SaveSlotsList.MODE_SAVE);
						return;
					case 1: // review mode
						goban.setMode (DrawingGoban.MODE_REVIEW);
						break;
					case 2: // play mode
						goban.setMode (DrawingGoban.MODE_PLAY);
						break;
					case 3: // score
						goban.setMode (DrawingGoban.MODE_COUNT);
						break;
					case 4: // change settings
						globalPreferences.initChangeSettings(d, null);
						return;
					case 5: // exit
						display.setCurrent(mainMenu);
						return;
				}
			}
			display.setCurrent(goban);
			return;
		} else if (d == sysinfo) {
			if (c == cmd_ok) display.setCurrent(mainMenu);
		}
	}

	

		
	private class SaveSlotsList extends List implements CommandListener {
		
		public static final int MODE_SAVE = 1;
		public static final int MODE_LOAD = 2;
		
		public static final int HTTP_CONNECTION_FAILED = 1;
		public static final int EGOBAN_TRANSFER_FAILED = 2;
				
		private boolean NO_RECORDSTORES=true;
		private int noOfRecordStores=0;
		
		private int mode = MODE_LOAD;
		
		private Displayable ret_screen;
		
		private DirectoryBrowser browser = new DirectoryBrowser();
		
		private Form saveNameForm = new Form("Save");
		private TextField saveNameField = new TextField("Save As", "", 42, TextField.ANY);
		
		private Form saveEgobanForm = new Form("Egoban");
		private TextField saveEgobanTextField = new TextField("Save As", "", 42, TextField.ANY);
		
		private Form transferResultForm = new Form("Server response");
		private StringItem response = new StringItem("Response:", null);
		
		
		
		private String nameToOverwrite="";
				
		public SaveSlotsList () {
			super("save slots",List.IMPLICIT);
		}
		
		public void invoke (Displayable ret_screen, int mymode)
		{
			this.ret_screen = ret_screen;
			setTitle(mymode == MODE_LOAD ? "Load game" : "Save game");

			if (mymode==MODE_SAVE) {
				int nextNoOfRecordStore=noOfRecordStores+1; 
				append("[Internally]",null);
				append("[Export as SGF]",null);
				append("[Export to egoban]", null);
			} 
			if (mymode == MODE_LOAD){
				append("[Import SGF]", null);
			}

			
			String [] availableRecordStores = RecordStore.listRecordStores();
			if (availableRecordStores!=null){
				NO_RECORDSTORES=false;
				noOfRecordStores=availableRecordStores.length;
				for(int i=0;i<noOfRecordStores;i++){
					if (!(
						(availableRecordStores[i].equals(EGOBAN_USER))
						|| (availableRecordStores[i].equals(EGOBAN_PASSWD))
						|| (availableRecordStores[i].equals(SHOW_HELP_AT_STARTUP))
						))
					append(availableRecordStores[i],null);
				}
			}
			
			this.mode = mymode;
			display.setCurrent(this);
			setCommandListener(this);
			if (!NO_RECORDSTORES) addCommand(cmd_select);
			addCommand(cmd_cancel);
		}
		
		public void loadFromRecordStore (String aktName)
		{
			try {		 
				RecordStore saveRecordStore = RecordStore.openRecordStore(aktName,false);
				brett = new GoBrett (saveRecordStore);
				goban.setMode(DrawingGoban.MODE_PLAY);
				goban.setBoardSize (brett.boardSize);
				saveRecordStore.closeRecordStore();
				
			} catch (RecordStoreException e){
				throw new RuntimeException("loadFromRecordStore failed: "+e.toString());
			}

		}
		
		public void initSaveToRecordStore() {
			String aktName = getString(getSelectedIndex());

			if (aktName!="[Internally]"){ // overwriting old recordstore
				nameToOverwrite = aktName;
			}
			else nameToOverwrite="";

			// call "Input page" here

			saveNameForm.append(saveNameField);
			if (aktName!="[Internally]"){
				StringItem warning = new StringItem("Warning!","Saving will overwrite game \""+nameToOverwrite+"\"");
				saveNameForm.append(warning);
			}
			saveNameForm.setCommandListener(this);
			saveNameForm.addCommand(cmd_ok);
			saveNameForm.addCommand(cmd_cancel);
			display.setCurrent(saveNameForm);
		}
		
		public void saveToRecordStore (String newName, String overwrittenRMS)
		{
			// if RS not empty: overwrite  (user should be asked for confirmation)
			if (overwrittenRMS!=null && overwrittenRMS!=""){
				try{
					RecordStore aktRMS = RecordStore.openRecordStore(overwrittenRMS,true);
					aktRMS.closeRecordStore();
					RecordStore.deleteRecordStore(overwrittenRMS);
					aktRMS = RecordStore.openRecordStore(newName,true);	
					brett.save(aktRMS);
					aktRMS.closeRecordStore();
				}
				catch (RecordStoreException e){
					throw new RuntimeException(e.toString());
				}
			}
			else{
				try {
					RecordStore aktRMS = RecordStore.openRecordStore(newName,true);
					brett.save(aktRMS);
					aktRMS.closeRecordStore();
				}
				catch (RecordStoreException e){
					throw new RuntimeException(e.toString());
				}			
			}	
		}
		
		public void saveToSGF (String path)
		{
			//security backup
			saveToRecordStore("AutoSave", "AutoSave");

			try {
				FileConnection conn = null;
				String url="file://"+path;
				
				if (path.endsWith(".sgf")) {
					conn = (FileConnection) Connector.open( url );
					// alert before overwriting is missing
					conn.delete();
				}
				else {
					int index = 1; 
					String newUrl = url+"mgo"+index+".sgf";
					conn = (FileConnection) Connector.open(newUrl);
					while (conn.exists()){
						//	check, which "mgo<index>.sgf" is not already used.
						index++;
						newUrl = url+"mgo"+index+".sgf";
						conn = (FileConnection) Connector.open(newUrl);
					}
				}
				conn.create();
				if (!conn.canRead()) conn.setReadable(true);
				if (!conn.canWrite()) conn.setWritable(true);
				
				String sequenceSgf = brett.toSGF();
				OutputStream out = conn.openOutputStream();
				PrintStream printStream = new PrintStream(out);
				//	now write data to the file
				
				printStream.print(sequenceSgf);
				printStream.flush();
				printStream.close();
				conn.close();
			}
			catch( Exception e ){
				System.out.println(e.toString());// error but no exception. This hopefully lets midlet continue
												// info screen (alert) is missing.
			}
			
		}
		
		public void loadFromSGF(String path)
		{
			try{
				FileConnection conn = null;
				String url="file://"+path;
				conn = (FileConnection) Connector.open( url);
				
				if (!conn.canRead()) conn.setReadable(true);
				
				InputStream is = conn.openInputStream();
				
				//restricted to 1024 byte (should be changeable in preferences)
				byte b[] = new byte[1024];
				int length = is.read(b, 0, 1024);
				String importedSGF=new String(b, 0, length);			
				conn.close();
				
				brett = new GoBrett (importedSGF);
				goban.setBoardSize (brett.boardSize);
				
				
			}
			catch( IOException e ){
				throw new RuntimeException(e.toString());
			}
			catch( SecurityException e ){
				throw new RuntimeException(e.toString());
			}
		}
		
		// this encoding was initially taken from a SUN java forum: 
		// http://forum.java.sun.com/thread.jspa?threadID=594204&messageID=3596899
		/**
		 * Encode a string according to W3C standards.
		 *
		 */
		public String urlEncoder(String s) {
			if (s == null)
				return s;
			StringBuffer sb = new StringBuffer(s.length() * 3);
			try {
				char c;
				for (int i = 0; i < s.length(); i++) {
					c = s.charAt(i);
					if (c == '&') {
						sb.append("&");
					} else if (c == ' ') {
						sb.append('+');
					} else if (
							(c >= 'A' && c <= 'Z')
							|| (c >= 'a' && c <= 'z')
							|| (c >='0' && c <='9')
							|| c == '_'
							|| c == '-'
							|| c == '.'
							|| c == '~'){
		
						sb.append(c);
					} else {
						sb.append('%');
						if (c > 15) { // is it a non-control char, ie. >x0F so 2 chars
							sb.append(Integer.toHexString((int)c)); // just add % and the string
						} else {
							sb.append("0" + Integer.toHexString((int)c));
							// otherwise need to add a leading 0
						}
					}
				}
	 
			} catch (Exception ex) {
				return (null);
			}
			return (sb.toString());
		}

		
		public void initSaveToEgoban(){
			
			if (globalPreferences.get(EGOBAN_USER) == Preferences.EMPTY // no username/password stored 
					|| globalPreferences.get(EGOBAN_PASSWD) == Preferences.EMPTY){ 
				globalPreferences.initChangeSettings(this, "Cannot save without credentials: enter here and try again.");
			}
			else{
				StringItem moneyWarning = new StringItem("Warning: ", 
						"Depending on your mobile provider, you will have to pay for data transfer." +
						"By pressing \"OK\", you agree to this.\n" +
						"If in doubt, press \"back\"! ");
				saveEgobanForm.append(moneyWarning);
				
				Date date = new Date();
				String actDate = date.toString();
				if (actDate.length()<25) actDate="";
				saveNameField = new TextField("Save As", actDate, 42, TextField.ANY);
				actDate = null;
				date = null;
				saveEgobanForm.append(saveNameField);
				
				StringItem isSavedAsRmsToo = new StringItem("Note:", "Game is additionally saved (\"AutoSave\"), in case something goes wrong.");
				saveEgobanForm.append(isSavedAsRmsToo);
				
				saveEgobanForm.setCommandListener(this);
				saveEgobanForm.addCommand(cmd_ok);
				saveEgobanForm.addCommand(cmd_cancel);
				display.setCurrent(saveEgobanForm);
			}
		}
		
		public void  showTransferResult(String result,int mode){
			if (mode == EGOBAN_TRANSFER_FAILED) {
				transferResultForm = new Form("Server response");
				response = new StringItem("egoban.org says: ", result);

				transferResultForm.append(response);
				if (!(result.equals("Success"))) 
					transferResultForm.append(new StringItem("Warning:", "Transfer seems to have failed. Check username and password in the settings."));
				else
					transferResultForm.append(new StringItem("Good News:", "Looks like the transfer went smoothly."));	
			}
			
			if (mode == HTTP_CONNECTION_FAILED) {
				transferResultForm = new Form("HTTP Error");
				response = new StringItem("Failure: ", result);
			}
			transferResultForm.setCommandListener(this);
			transferResultForm.addCommand(cmd_ok);	
			display.setCurrent(transferResultForm);
		}
		
		public void saveToEgoban(String sgf,String gameTitle){
			
			//security backup
			saveToRecordStore("AutoSave", "AutoSave");
			
			saveEgobanForm.append(saveEgobanTextField);
			saveEgobanForm.setCommandListener(this);
			saveEgobanForm.addCommand(cmd_ok);
			saveEgobanForm.addCommand(cmd_cancel);
			display.setCurrent(saveEgobanForm);
			
			
			//System.out.println("SGF to be sent to egoban: "+sgf);
						
			String url = "http://egoban.org/script_mgoUpload?gameTitle="+urlEncoder(gameTitle)+"&sgfText="+urlEncoder(sgf);
			
			byte[] loginByteArray = (globalPreferences.get(EGOBAN_USER)+":"+globalPreferences.get(EGOBAN_PASSWD)).getBytes();
			String base64credentials = new String(Base64.encode(loginByteArray));
			try{
				HttpConnection connection = (HttpConnection) Connector.open(url, Connector.READ_WRITE);
				connection.setRequestMethod(HttpConnection.GET);
				connection.setRequestProperty("Authorization","Basic " + base64credentials);
				int responseCode = connection.getResponseCode();
				//System.out.println("Response code: "+responseCode);
				
				if (responseCode == 200) //successful HTTP connection
				{
					InputStream inputStream = connection.openInputStream();
					StringBuffer responseBuffer = new StringBuffer();
					
					int responseCharacter=0;
					for (int i = 0; i<100 && responseCharacter != -1; i++) {
						responseCharacter=inputStream.read();
						if (responseCharacter != -1)
						responseBuffer.append((char)responseCharacter);
					}   			
					connection.close();

					//System.out.println("response stringbuffer: "+responseBuffer);
					showTransferResult(responseBuffer.toString(), EGOBAN_TRANSFER_FAILED);
				}
				else{
					showTransferResult("HTTP connection failed.", HTTP_CONNECTION_FAILED);		
					}
				
				
			}
			catch (Exception e){
				System.out.println(e.toString());
			}
		}
		
		public void commandAction (Command c, Displayable d)
		{
			String selection = getString(getSelectedIndex());
			if (c == cmd_cancel) {
				display.setCurrent(ret_screen);
			} 
			if ((c == cmd_select || c == SELECT_COMMAND)) {
				if (this.mode == MODE_LOAD) {
					if (selection=="[Import SGF]") {
						browser.invoke(this,this);
						return;
					}
					else
						if (!NO_RECORDSTORES) 
							loadFromRecordStore (getString(getSelectedIndex()));
					display.setCurrent(goban);
				} 
				if (this.mode == MODE_SAVE) {
					if (selection == "[Export as SGF]") { // sgf export
						browser.invoke(this,this);
						return;
					} 
					else if (selection == "[Export to egoban]") {// online export
						initSaveToEgoban();
						return;
					}
					else 
						initSaveToRecordStore ();
				}	
			} 
			if (c == DirectoryBrowser.SUBMIT_COMMAND) { //(d == browser &&
				if (this.mode == MODE_SAVE)	
					saveToSGF (browser.getPath());
				if (this.mode == MODE_LOAD)
					loadFromSGF(browser.getPath());

				display.setCurrent(goban);
			}
			if (d==saveNameForm && c==cmd_ok) { //Save As{
				String newName = saveNameField.getString();
				saveToRecordStore(newName, nameToOverwrite); //if nameToOverwrite is null: new, else:overwrite
				display.setCurrent(goban); 
				return;
			}	
			if (d==saveEgobanForm && c==cmd_ok) {
				saveToEgoban(brett.toSGF(),saveNameField.getString());
				return;
			}
			if (d==transferResultForm && c==cmd_ok){
				display.setCurrent(ret_screen);
				return;
			}
		} // commandAction
	}

	private static class DirectoryBrowser extends List implements CommandListener {
		
		private Displayable ret_screen;
		private CommandListener next_screenCommandListener;
		private String path = "/";
		
		private static Command cmd_cancel = new Command("Back", Command.CANCEL, 2);
		private static Command cmd_up = new Command("Up", Command.BACK, 2);
		private static Command cmd_select = new Command("Select", Command.ITEM, 1);
		private static Command cmd_ok = new Command("OK", Command.OK, 1);
		
		private static Command cmd_delete = new Command ("Delete", Command.ITEM, 2);
		
		public static Command SUBMIT_COMMAND = cmd_select;
		
		public DirectoryBrowser () { super("Directory browser",List.IMPLICIT); }
		
		public void invoke (Displayable myReturnScreen, CommandListener nextCommandListener) {
			ret_screen = myReturnScreen;
			next_screenCommandListener = nextCommandListener;
			display.setCurrent(this);
			setCommandListener(this);
			addCommand(cmd_cancel);
			addCommand(cmd_up);
			addCommand(cmd_select);
			addCommand(cmd_delete);
			setDirectory (path);
		}
				
		public void setDirectory (String target) {
			if ("..".equals(target)) {
				leaveDirectory();
				return;
			}
			
			if (target.charAt(0) == '/') // absolute path
				path = target;
			else // one directory deeper
				path += "" + target;
			
			while (size() > 0) delete(0); // clear all ... deleteAll() is in MIDP2.0
			
			if (!"/".equals(path)){
				append(". (this dir)",null);
				append("..",null);
			}
			
			// actual directory listing 
			if ("/".equals(path)){
				// get root dirs
				try {
					//System.err.println("opening roots");
					Enumeration theRoots=FileSystemRegistry.listRoots();
					//System.err.println("roots opened");
					
					while (theRoots.hasMoreElements()) {
						String element = (String) theRoots.nextElement();       
						append(element,null);
					}
				}	
				catch( SecurityException e ){
					throw new RuntimeException(e.toString());// no permission to read/write
				}
			}
			else{
				if (path.endsWith("/")){
				// get files and directories
				
				//TODO dirs should be marked
				try {
					FileConnection conn=null;
					conn = (FileConnection) Connector.open("file://"+path);
					Enumeration fileList = conn.list(); // files and directories
					String nextFile=null;
					while (fileList.hasMoreElements()){
						nextFile = (String) fileList.nextElement();
						//ToDO path extension and leaveDirectory are not in sync. 
						// it needs two leaveDirectory to actually leave it.
						//append(path+"/"+nextFile, null);
						append(path+nextFile, null);
					}
					conn.close();
				}
				catch( IOException e ){
					throw new RuntimeException(e.toString());// error
				}
				catch( SecurityException e ){
					throw new RuntimeException(e.toString());// no permission to create/write
				}
				}
			}
			setTitle("dir: "+path);
		}
		
		public void leaveDirectory () {
			if ("/".equals(path)) return;
			int lastSlashIndex=path.lastIndexOf('/',path.length()-2);
			path=path.substring(0,lastSlashIndex+1);
			setDirectory (path);
		}
		
		public void commandAction (Command c, Displayable d) {
			if (c == cmd_cancel) {
				display.setCurrent(ret_screen);
				return;
			} else if (c == cmd_up)  {
				// jump one level below
				leaveDirectory();
			} else if (c == cmd_select || c == SELECT_COMMAND) {
				if (getString(getSelectedIndex()).equals(". (this dir)")){
					// if actual dir is selected
					next_screenCommandListener.commandAction(SUBMIT_COMMAND,this);
					return;
				}
				else if (getString(getSelectedIndex()).endsWith(".sgf")) {
					this.setPath(getString(getSelectedIndex()));
					next_screenCommandListener.commandAction(SUBMIT_COMMAND,this);
					return;
				}
				else {
					// jump into selected (distinct treatment of directories 
					// and files lies in setDirectory() 
					setDirectory (getString(getSelectedIndex()));			
				}
			} 
			else if (c == cmd_delete){
				try {
					FileConnection conn = (FileConnection) Connector.open( "file://"+getString(getSelectedIndex()) );
					// 	alert before overwriting is missing
					conn.delete();
					conn.close();
					//works, but display is not renewed
					setDirectory(path);
					return;
				}
				catch( IOException e ){
					throw new RuntimeException(e.toString());// error
				}
				catch( SecurityException e ){
					throw new RuntimeException(e.toString());// no permission to create/write
				}
			}
			display.setCurrent(this);
		}

		public void setPath (String mypath) { this.path = mypath; }
		public String getPath () { return path; }
	}
	
	private class SysInfoScreen extends Canvas {
		public void paint (Graphics g) {
			int hLength = this.getWidth();
			int vLength = this.getHeight();

			g.setGrayScale(255);
			g.fillRect(0,0,hLength,vLength);
			g.setGrayScale(0);
			Font font= Font.getFont(Font.FACE_SYSTEM, Font.STYLE_BOLD, Font.SIZE_MEDIUM);
			g.drawString("SysInfo", hLength/2, 0, Graphics.TOP | Graphics.HCENTER);
			font=Font.getDefaultFont();
			g.drawString("w:"+hLength+", h:"+vLength, 0, 1*font.getHeight(), Graphics.TOP	| Graphics.LEFT);
			
			
			String fileConnectionInfo="";
			if (fileConnectionVersion != null){
				fileConnectionInfo= "yes";
			}
			else {
				fileConnectionInfo= "no";
			}
			g.drawString("export: "+fileConnectionInfo, 0, 2*font.getHeight(), Graphics.TOP	| Graphics.LEFT);			
			int lineNo=vLength/font.getHeight();
			g.drawString("lines: "+lineNo, 0, 3*font.getHeight(), Graphics.TOP	| Graphics.LEFT);
            g.drawString(cvs_Id, 0, 4*font.getHeight(), Graphics.TOP  | Graphics.LEFT);
            if (globalPreferences.get(EGOBAN_USER) == Preferences.EMPTY)
            	g.drawString("Egoban username: not set", 0, 5*font.getHeight(), Graphics.TOP | Graphics.LEFT);
            else
            	g.drawString("Egoban username: set", 0, 5*font.getHeight(), Graphics.TOP | Graphics.LEFT);
            if (globalPreferences.get(EGOBAN_PASSWD) == Preferences.EMPTY)
            	g.drawString("Egoban password: not set", 0, 6*font.getHeight(), Graphics.TOP | Graphics.LEFT);
            else
            	g.drawString("Egoban username: set", 0, 6*font.getHeight(), Graphics.TOP | Graphics.LEFT);

		}
	}
	
	private class HelpScreen extends Form implements CommandListener {
		private Displayable ret_screen;
		private StringItem content;
		
		private Command cmd_ok = new Command("OK", Command.OK, 1);
		
		public HelpScreen () {
			super("help");
			content = new StringItem("help", "if you see this, the developer screwed up");
			append(content);
			setCommandListener(this);
			addCommand(cmd_ok);
		}
		
		public void invoke (Displayable back, String title, String contents)
		{
			content.setLabel(title);
			content.setText(contents);
			ret_screen = back;
			display.setCurrent(this);
		}
		
		public void commandAction (Command c, Displayable d)
		{
			if (c == cmd_ok) {
				display.setCurrent(ret_screen);
			}
		}
	}
	
	private class ScoreScreen extends Canvas {
		int captivesBlack, captivesWhite;
		int pointsBlack, pointsWhite;
		
		public void paint (Graphics g) {
			int hLength = getWidth();
			int vLength = getHeight();
			
			//	clear screen
			g.setGrayScale(255);
			g.fillRect(0,0,hLength,vLength);
		
			g.setGrayScale(0);
			
			Font font=Font.getFont(Font.FACE_SYSTEM, Font.STYLE_BOLD, Font.SIZE_MEDIUM);
			
			g.drawString("Score",hLength/2,0,Graphics.TOP|Graphics.HCENTER);
			
			font = Font.getDefaultFont();
							
			g.drawString("Black",0,font.getHeight(),Graphics.TOP|Graphics.LEFT);
			g.drawString("White",hLength,font.getHeight(),Graphics.TOP|Graphics.RIGHT);
			
			g.drawString("Pts",hLength/2,2*font.getHeight(),Graphics.TOP|Graphics.HCENTER);
			
			g.drawString(Integer.toString(pointsBlack),0,2*font.getHeight(),Graphics.TOP|Graphics.LEFT);
			g.drawString(Integer.toString(pointsWhite),hLength,2*font.getHeight(),Graphics.TOP|Graphics.RIGHT);
			
			
			g.drawString("Cpt",hLength/2,3*font.getHeight(),Graphics.TOP|Graphics.HCENTER);
			
			g.drawString(Integer.toString(captivesBlack),0,3*font.getHeight(),Graphics.TOP|Graphics.LEFT);				
			g.drawString(Integer.toString(captivesWhite),hLength,3*font.getHeight(),Graphics.TOP|Graphics.RIGHT);
			
			g.drawString("Sum.",hLength/2,4*font.getHeight(),Graphics.TOP|Graphics.HCENTER);
			
			g.drawString(Integer.toString(captivesBlack+pointsBlack),0,4*font.getHeight(),Graphics.TOP|Graphics.LEFT);				
			g.drawString(Integer.toString(captivesWhite+pointsWhite),hLength,4*font.getHeight(),Graphics.TOP|Graphics.RIGHT);
		}
	}
	
	private class CaptsScreen extends Canvas implements CommandListener {
		public int blackCapts;
		public int whiteCapts;
		
		public int mode;
		
		private Command cmd_ok = new Command("OK", Command.OK, 1);
		
		public void paint (Graphics g) {
			int hLength = getWidth();
			int vLength = getHeight();
			
			//	clear screen
			g.setGrayScale(255);
			g.fillRect(0,0,hLength,vLength);
		
			g.setGrayScale(0);
			
			Font font=Font.getFont(Font.FACE_SYSTEM, Font.STYLE_BOLD, Font.SIZE_MEDIUM);
			
			g.drawString("Captures",hLength/2,0,Graphics.TOP|Graphics.HCENTER);
			
			font = Font.getDefaultFont();
							
			g.drawString("Black",0,font.getHeight(),Graphics.TOP|Graphics.LEFT);
			g.drawString("White",hLength,font.getHeight(),Graphics.TOP|Graphics.RIGHT);
			
			g.drawString(""+blackCapts,0,2*font.getHeight(),Graphics.TOP|Graphics.LEFT);
			g.drawString(""+whiteCapts,hLength,2*font.getHeight(),Graphics.TOP|Graphics.RIGHT);
						
			setCommandListener(this);
		    addCommand(cmd_ok);
				}
		
		public void commandAction(Command c, Displayable d){
			if (c == cmd_ok){
				goban.setMode (mode);
				display.setCurrent(goban);
				repaint();
			}
		}
		
	}
	
	private class LogoScreen extends Canvas implements CommandListener{
		Image logo;
		int width=getWidth();
		int height=getHeight();
		String imagePath;
		boolean logoAvailable=false;
				
		public LogoScreen(){
//			if ( width >= 200 && height >=62 ) imagePath="mgo.logo.200x62.transparent.png";
//			else if (width >=120 && height >=36) imagePath="mgo.logo.120x36.transparent.png";
//			else if (width >=80 && height >=27) imagePath="mgo.logo.button.88x27.transparent.png";
//			//else if (width >=80 && height >=27) imagePath="mgo.256.png";
//			else if (width >=60 && height >=18) imagePath="mgo.logo.60x18.transparent.png";
//			else if (width >=40 && height >=12) imagePath="mgo.logo.40x12.transparent.png";
//			
//			try {
//				logo=Image.createImage("/"+imagePath);
//				logoAvailable=true;
//			}
//			catch (IOException e){
//				logoAvailable=false;
//				throw new RuntimeException("Unable to load Image "+imagePath+": " +e);
//			}
			setCommandListener(this);
			addCommand(cmd_ok);
			addCommand(cmd_cancel);
		}
		

		
		
		public void paint (Graphics g){
			
			// deliberately set off, as graphic has aesthetical problems on small displays
			logoAvailable=false;
			
			// clear screen
			g.setGrayScale(255);
			g.fillRect(0,0,getWidth(),getHeight());
			
						
			if (logoAvailable) g.drawImage(logo, getWidth()/2, getHeight()/2, Graphics.HCENTER | Graphics.VCENTER);
			//Font font=Font.getDefaultFont();
			g.setGrayScale(0);
			g.drawString("www.mobilego.org", width/2, height/2, Graphics.BOTTOM | Graphics.HCENTER);
		}
		
		public void commandAction(Command c, Displayable d){
			if (c == cmd_ok){
				display.setCurrent(mainMenu);
			}
			if (c== cmd_cancel){
				destroyApp(false);
				notifyDestroyed();
			}
		}
	}

	private class DrawingGoban extends Canvas implements CommandListener {
		
		private Command cmd_pass = new Command ("Pass", Command.SCREEN, 1);
		private Command cmd_undo = new Command ("Undo", Command.SCREEN, 2);
		private Command cmd_score = new Command ("Count", Command.SCREEN, 1);
		private Command cmd_capts = new Command ("Capts", Command.SCREEN,1);
		private Command cmd_menu = new Command ("Options", Command.BACK, 3);
		private Command cmd_help = new Command ("Help", Command.HELP, 4);
		
		public static final String help_play = "\n\n" +
			"#: Pass\n" +
			"*: Take back\n" +
			"\n" +
			"move cursor around with direction keys or numbers 1-9, " +
			"place stone with Fire or number 5\n";

		public static final String help_count = "\n\n" +
			"move cursor around with direction keys or numbers 1-9, " +
			"remove group under cursor with Fire or number 5\n";
		
		public static final String help_review = "\n\n" +
			"4/6: back / forward\n" +
			"8/2: -5 / +5 moves\n" +
			"1/3: -10 / +10 moves\n" +
			"7/9: start / end\n" +
			"\n" +
			"you can also use direction keys to move back, forward (left, right) " +
			"or 5 moves back and forward (down, up)";

		int boardSize = 9;
		int[] hoshis;
		
		Coords cursor = new Coords();
		
		public static final int MODE_PLAY = 1;
		public static final int MODE_REVIEW = 2;
		public static final int MODE_COUNT = 3;
		private int mode;
		
		private static final int minHoshiSize = 3;

		GoBrett countBrett;
		Stack freeGroups = new Stack();

		// number of black's and white's territority points
		int blacks, whites;
		
		// size of the grid in pixels
		int hLength, vLength;
		// horiz/vert spacing in pixels, smaller of the two
		int hSpace, vSpace, minSpace;
		// size of lastmove and ko-mark in pixels
		int markSize;
		
		// pre-rendered images of grid, black/white stone, black/white territority mark
		Image board, black, white, mark_black, mark_white;
		
		public DrawingGoban () {
			score.setCommandListener(this);
			score.addCommand(cmd_menu);
			
			setCommandListener(this);
			addCommand(cmd_menu);
			addCommand(cmd_help);
			
			setMode(MODE_PLAY);
		}
		
		public void setHoshiIndex (int hoshiIndex)
		{
			boardSize = boardSizes[hoshiIndex];
			hoshis = boardHoshis[hoshiIndex];
			
			init();
		}
		
		public void setBoardSize (int boardSize)
		{
			this.boardSize = boardSize;
			// find matching hoshis record
			this.hoshis = null;
			for (int i = 0; i < boardSizes.length; i++)
				if (boardSizes[i] == boardSize) {
					hoshis = boardHoshis[i];
					break;
				}
			
			init();
		}
		
		public void init ()
		{
			cursor = new Coords();
			// screen dimensions
			hLength = this.getWidth();
			vLength = this.getHeight();
			
			// don't let it become to much non-square (here: ration not > 1.3)

			if (vLength*10 > hLength*13){
				vLength=(13*hLength)/10;
			}
			if (hLength*10 > vLength*13){
				hLength=(13*vLength)/10;
			}
			
			// space between lines
			hSpace = hLength/(boardSize+1);
			vSpace = vLength/(boardSize+1);
			
			minSpace = Math.min(hSpace,vSpace);
			
			int stoneSize=minSpace;
			
			int terrSize = Math.max(2,stoneSize/2);
						
			// try to get mark as long as stone radius
			markSize=((stoneSize/2)*707)/1000;
			//if ((markSize)<2) markSize=0;	
			
			// pre-render stones (drawing circles is expensive)
			white = Image.createImage(stoneSize+1,stoneSize+1);
			Graphics g = white.getGraphics();
			g.setGrayScale(255); 
			g.fillArc(0, 0, stoneSize, stoneSize, 0, 360); 
			g.setGrayScale(0);
			g.drawArc(0, 0, stoneSize, stoneSize, 0, 360);
			
			black = Image.createImage(stoneSize+1, stoneSize+1);
			g = black.getGraphics();
			g.setGrayScale(0);
			g.fillArc(0, 0, stoneSize+1, stoneSize+1, 0, 360);
			
			// pre-render marks
			mark_white = Image.createImage(terrSize+1, terrSize+1);
			g = mark_white.getGraphics();
			g.setGrayScale(255);
			g.fillArc(0, 0, terrSize, terrSize, 0, 360);
			g.setGrayScale(0);
			g.drawArc(0, 0, terrSize, terrSize, 0, 360);
			
			mark_black = Image.createImage(terrSize+1, terrSize+1);
			g = mark_black.getGraphics();
			g.setGrayScale(0);
			g.fillArc(0, 0, terrSize+1, terrSize+1, 0, 360);

			// pre-render board
			board = Image.createImage(hLength, vLength);
			drawGoban(board.getGraphics());
		}
		
		private void drawGoban (Graphics g) {
			// clear screen
			g.setGrayScale(255);
			g.fillRect(0,0,hLength-1,vLength-1);
			
			// draw goban (i.e. horizontal and vertical lines)
			g.setGrayScale(0);
			for (int i=1; i<=boardSize; i++) {
				g.drawLine(i*hSpace, 1*vSpace, i*hSpace, boardSize*vSpace);
				g.drawLine(1*hSpace, i*vSpace, boardSize*hSpace, i*vSpace);
			}
			
			// draw hoshis
			if (hoshis != null) {
				int hoshiSize=(minSpace*3)/6;
				if (hoshiSize<minHoshiSize) hoshiSize=2;
				int hoshiVersatz=hoshiSize/2;
			
				g.setGrayScale(75);
				for (int x = 0; x < hoshis.length; x++)
					for (int y = 0; y < hoshis.length; y++) {
						if (hoshiSize>=minHoshiSize){
							g.drawArc(hoshis[x]*hSpace-hoshiVersatz, hoshis[y]*vSpace-hoshiVersatz, 
								hoshiSize, hoshiSize, 
								0, 360);
						} else {
							g.fillArc(hoshis[x]*hSpace-hoshiVersatz, hoshis[y]*vSpace-hoshiVersatz, 
								hoshiSize, hoshiSize, 
								0, 360);
						}
					}
			}
		}
		
		private void drawStones (Graphics g, GoBrett brett, boolean markLastMove) {
			
			Coords lastmove = brett.lastMove();
			// To speed things up, we'll walk through entire board:
			// vvalue and hvalue are vertical (horizontal) offsets for current
			// intersection on screen. At each step, we add one space
			// to one of the offsets.
			// This way we use only trivial math in linear time.
			
			int vvalue = vSpace;
			for (int j=0; j<boardSize; j++) { // note that j (y) goes in outer cycle!
				int hvalue = hSpace;
				for (int i=0; i<boardSize; i++) {
					if (brett.contains(i,j)) {
						int col2 = 128;
						switch (brett.get(i,j)) {
							case GoBrett.WHITE:
								g.drawImage(white, hvalue, vvalue, Graphics.HCENTER | Graphics.VCENTER);
								col2 = 0;
								break;
							case GoBrett.BLACK:
								g.drawImage(black, hvalue, vvalue, Graphics.HCENTER | Graphics.VCENTER);
								col2 = 255;
								break;
							case GoBrett.MARK_WHITE: if (mode == MODE_COUNT)
								g.drawImage(mark_white, hvalue, vvalue, Graphics.HCENTER | Graphics.VCENTER);
								break;
							case GoBrett.MARK_BLACK: if (mode == MODE_COUNT)
								g.drawImage(mark_black, hvalue, vvalue, Graphics.HCENTER | Graphics.VCENTER);
								break;
						}
						
						// mark last move
						if (markLastMove && lastmove!=null && lastmove.equals(i,j)) {
							g.setGrayScale(col2);
							g.drawLine(hvalue-markSize,
									vvalue-markSize,
									hvalue+markSize,
									vvalue+markSize); 
							g.drawLine(hvalue+markSize,
									vvalue-markSize,
									hvalue-markSize,
									vvalue+markSize);
						}	
					}
					if (brett.koPosition!=null && brett.koPosition.equals(i,j)){
						//print KO-marker
						g.setGrayScale(0);
						
						g.drawLine(hvalue-markSize, 
								vvalue-markSize, 
								hvalue+markSize,
								vvalue+markSize); 
						g.drawLine(hvalue+markSize, 
								vvalue-markSize, 
								hvalue-markSize,
								vvalue+markSize);
						
					}
					hvalue += hSpace;
				}
				vvalue += vSpace;
			}
		}
		
		private void drawCursor (Graphics g) {
			int cursorSize=(minSpace*3)/2;
			int cursorVersatz=cursorSize/2;
			
			if (brett.color == GoBrett.BLACK) g.setColor(0,0,255);
			else g.setColor(255,0,0);
			
			g.drawRect((cursor.x+1)*hSpace-cursorVersatz,
					(cursor.y+1)*vSpace-cursorVersatz,
					cursorSize,cursorSize);
		}
		
		public void paint (Graphics g) {
			//drawGoban (g);
			g.drawImage(board, 0, 0, 0);
			
			switch (getMode()) {
				case MODE_PLAY:
					drawStones (g, brett, true);
					drawCursor(g);
					break;
				case MODE_REVIEW:
					drawStones (g, brett, true);
					// no cursor for review mode
					break;
				case MODE_COUNT:
					drawStones (g, countBrett, false);
					drawCursor(g);
					break;
			}
		}

		public void c_up () { cursor.y--; if (cursor.y < 0) cursor.y = boardSize-1; }
		public void c_down () { cursor.y++; if (cursor.y >= boardSize) cursor.y = 0; }
		public void c_left () { cursor.x--; if (cursor.x < 0) cursor.x = boardSize-1; }
		public void c_right () { cursor.x++; if (cursor.x >= boardSize) cursor.x = 0; }
		
		private boolean handleCommonKeys (int keyCode)
		{
			switch (keyCode) {
				case KEY_NUM0:
					display.setCurrent(ingameMenu);
					break;
				default:
					return false;
			}
			return true;
		}

		private void handleFireKey ()
		{
			switch (mode) {
			case MODE_PLAY:
				int moveResult = brett.play(cursor);
				if (moveResult==GoBrett.M_NORMAL) repaint();
				break;
				
			case MODE_COUNT:
				if ((countBrett.get(cursor)==GoBrett.BLACK) || (countBrett.get(cursor)==GoBrett.WHITE)){
					countBrett.removeGroup(cursor);
					recountPoints();
					repaint();
				}
				break;
			case MODE_REVIEW:
				break;
			}
		}
		
		private boolean handleCursorKeys (int keyCode)
		{
			switch (getGameAction(keyCode)) {
				case UP:
					c_up ();
					break;
				case DOWN: 
					c_down ();
					break;
				case LEFT: 
					c_left ();
					break;
				case RIGHT: 
					c_right ();
					break;
				case FIRE:
					handleFireKey();
				default:
					switch (keyCode) {
						case KEY_NUM3:
							c_right(); c_up();
							break;
						case KEY_NUM1:
							c_left(); c_up();
							break;
						case KEY_NUM7:
							c_left(); c_down();
							break;
						case KEY_NUM9:
							c_right(); c_down();
							break;
						default:
							return false;
					}
			}
			repaint();
			return true;
		}

		private boolean handlePlayKeys (int keyCode)
		{
			if (handleCursorKeys(keyCode)) return true;
			
			switch (keyCode) {
				case KEY_STAR: // undo
					brett.undo();
					repaint();
					break;
				case KEY_POUND: // pass
					brett.pass();
					repaint();
					break;
				default:
					return false;
			}
			return true;
		}

		private boolean handleCountKeys (int keyCode)
		{
			if (handleCursorKeys(keyCode)) return true;
			
			if (keyCode == KEY_POUND) {
				score.pointsBlack = blacks;
				score.pointsWhite = whites;
				score.captivesBlack = countBrett.captives[GoBrett.BLACK];
				score.captivesWhite = countBrett.captives[GoBrett.WHITE];
				display.setCurrent(score);
				return true;
			} else
				return false;
		}

		private boolean handleReviewKeys (int keyCode)
		{
			switch (getGameAction(keyCode)) {
				// show previous move (go back in history)
				case LEFT: 
					brett.back();
					break;
				// show next move (go ahead in review)
				case RIGHT:
					brett.forward();
					break;
				// 5 moves back
				case UP:
					brett.back(5);
					break;
				// 5 moves forward
				case DOWN:
					brett.forward(5);
					break;
				default:
					switch (keyCode) {
						case KEY_NUM7: // to first move
							while (brett.back()) ;
							break;
						case KEY_NUM9: // to last move
							while (brett.forward()) ;
							break;
						case KEY_NUM1: // 10 moves back
							brett.back(10);
							break; 
						case KEY_NUM3: // 10 moves ahead
							brett.back(10);
							break;
						default:
							return false;
					}
			}
			repaint();
			return true;
		}
		
		public void keyPressed(int keyCode) {
			if (handleCommonKeys(keyCode)) return;
			
			switch (getMode()) {
				case MODE_PLAY:
					handlePlayKeys (keyCode);
					break;
				case MODE_REVIEW:
					handleReviewKeys (keyCode);
					break;
				case MODE_COUNT:
					handleCountKeys (keyCode);
					break;
			}
		}

		public void keyRepeated(int keyCode) {
			keyPressed(keyCode);
		}
		
		private boolean putXYOnBoard (int cx, int cy) {
			cx -= hSpace/2;
			cy -= vSpace/2;
			if (cx >= 0 && cx <= hLength-hSpace && cy >= 0 && cy <= vLength-vSpace) {
				cursor.x = cx/hSpace;
				cursor.y = cy/vSpace;
				return true;
			}
			return false;
		}
		
		public void pointerPressed (int cx, int cy) {
			if (putXYOnBoard(cx,cy)) repaint();
		}
		
		public void pointerDragged (int x, int y) { pointerPressed(x,y); }
		
		public void pointerReleased (int cx, int cy) {
			if (putXYOnBoard(cx,cy)) {
				handleFireKey();
				repaint();
			}
		}
		
		public void commandAction (Command c, Displayable d)
		{ 
			if (c == cmd_pass && mode == MODE_PLAY) {
				brett.pass();
				repaint();
			} else if (c == cmd_undo) {
				if (mode == MODE_PLAY) {
					brett.undo();
					repaint();
				} else if (mode == MODE_COUNT) {
					// TODO undo group delete
				}
			} else if (c == cmd_score && mode == MODE_COUNT) {
				score.pointsBlack = blacks;
				score.pointsWhite = whites;
				score.captivesBlack = countBrett.captives[GoBrett.BLACK];
				score.captivesWhite = countBrett.captives[GoBrett.WHITE];
				display.setCurrent(score);
			} else if (c == cmd_capts){
				capts.blackCapts=brett.captives[GoBrett.BLACK];
				capts.whiteCapts=brett.captives[GoBrett.WHITE];
				capts.mode = mode;
				display.setCurrent(capts);
			} else if (c == cmd_menu) {
				display.setCurrent(ingameMenu);
			} else if (c == cmd_help) {
				String text, title;
				if (goban.getMode() == DrawingGoban.MODE_PLAY) {
					title = "Game Controls";
					text = DrawingGoban.help_play;
				} else if (goban.getMode() == DrawingGoban.MODE_REVIEW) {
					title = "Review Controls";
					text = DrawingGoban.help_review;
				} else {
					title = "Counting Controls";
					text = DrawingGoban.help_count;
				}
				help.invoke(this,title,text);
			}
		}
		
		private void recountPoints ()
		{
			// cleanup marked groups from previous counting
			while (! freeGroups.isEmpty()) {
				Vector group = (Vector)freeGroups.pop();
				countBrett.removeStones(group);
			}
			
			// walk through whole board...
			for (int i = 0; i < boardSize; i++) for (int j = 0; j < boardSize; j++) {
				if (countBrett.empty(i,j)) { // if unmarked point is found...
					// find the whole group
					Vector group = countBrett.getGroup(new Coords(i,j));
					freeGroups.push (group);
					// and mark it
					countBrett.putStones(group, GoBrett.MARK_NEUTRAL);
				}
			}
			
			blacks = whites = 0;

			// decide ownership of found groups
			for (int i = 0; i < freeGroups.size(); i++) {
				Vector group = (Vector)freeGroups.elementAt(i);
				boolean black = false;
				boolean white = false;
				for (int j = 0; j < group.size(); j++) {
					// check neighbours of all stones
					Coords stone = (Coords)group.elementAt(j);
					if (countBrett.countTouching(stone,GoBrett.BLACK)>0) black = true;
					if (countBrett.countTouching(stone,GoBrett.WHITE)>0) white = true;
					if (black && white) break; // ownership can't be decided, look no further
				}
				if (black && !white) { // territority belongs to black
					countBrett.putStones(group,GoBrett.MARK_BLACK);
					blacks += group.size();
				}
				else if (white && !black) { // ...or to white
					countBrett.putStones(group,GoBrett.MARK_WHITE);
					whites += group.size();
				}
			}			
		}
		
		private void prepareCounting ()
		{
			brett.endReview();
			removeCommand(cmd_pass);
			removeCommand(cmd_undo);
			addCommand(cmd_score);
			
			// copy of brett.data
			countBrett = new GoBrett(brett);
			freeGroups.removeAllElements(); 
			
			recountPoints();
		}
		
		private void prepareReview ()
		{
			removeCommand(cmd_pass);
			removeCommand(cmd_undo);
			removeCommand(cmd_score);

			brett.beginReview();
		}
		
		private void preparePlay ()
		{
			if (brett!=null) brett.endReview();
			// XXX this check should be avoided, right?
			removeCommand(cmd_score);
			addCommand(cmd_pass);
			addCommand(cmd_undo);
			addCommand(cmd_capts);
		}
		
		public void setMode (int mode)
		{
			if (mode != this.mode) {
				switch (mode) {
					case MODE_REVIEW:
						prepareReview();
						break;
					case MODE_PLAY:
						preparePlay();
						break;
					case MODE_COUNT:
						prepareCounting();
						break;
				}
			}
			this.mode = mode;
		}

		public int getMode () { return mode; }

	}

	public static class Coords {
		public int x;
		public int y;
		public Coords () { x = 0; y = 0; }
		public Coords (int x, int y) { this.x = x; this.y = y; }
		public Coords (char x, char y) { this.x = x - 'a'; this.y = y - 'a'; }
		public Coords (Coords c) { this.x = c.x; this.y = c.y; }
		
		public boolean equals (Coords c) { return c != null && x == c.x && y == c.y; }
		public boolean equals (int x, int y) { return this.x == x && this.y == y; }
		public String toSGF () { return "" + (char)('a'+x) + (char)('a'+y); }
		public byte bx () { return (byte)x; }
		public byte by () { return (byte)y; }
	}
	
	private class Preferences implements CommandListener{
		/* 
		 * Settings are stored by RMS.
		 * This class implements methods to read/write the settings/preferences
		 * 
		 */
		public static final String EMPTY = "-1";
		public static final String DO_SHOW_HELP = "0";
		
		Form settingsForm = new Form("Settings");
		private TextField egobanUsernameField = new TextField("Egoban Usernamame:", "", 42, TextField.ANY);
		private TextField egobanPasswordField = new TextField("Egoban Password:", "", 42, TextField.ANY);
		private ChoiceGroup startupHelpScreen = new ChoiceGroup("Show Help at Start?", Choice.EXCLUSIVE);
		Displayable ret_screen;
		
		public Preferences() {
			try{
			    RecordStore settingsRMS = RecordStore.openRecordStore(EGOBAN_USER,true);
			    settingsRMS.closeRecordStore();
			    settingsRMS = RecordStore.openRecordStore(EGOBAN_PASSWD,true);
			    settingsRMS.closeRecordStore();
			}
			catch (RecordStoreException e){
				throw new RuntimeException(e.toString());
			}
		
		}
		
		private void set(String parameter, String value){	
			try {
				RecordStore settingsRMS = RecordStore.openRecordStore(parameter,true);
				if (settingsRMS.getNumRecords()>0){
					settingsRMS.closeRecordStore();
					RecordStore.deleteRecordStore(parameter);
					settingsRMS = RecordStore.openRecordStore(parameter,true);
				}
				settingsRMS.addRecord(value.getBytes(), 0, value.getBytes().length);
				settingsRMS.setRecord(1,value.getBytes(), 0, value.getBytes().length);
				//byte[] content = settingsRMS.getRecord(1);
				settingsRMS.closeRecordStore();
			}
			catch (RecordStoreException e){
				throw new RuntimeException(e.toString());
			}

		}
		private String get(String parameter){
			String result = null;
			try {
				RecordStore settingsRMS = RecordStore.openRecordStore(parameter,true);
				if (settingsRMS.getNumRecords() > 0) {
					if (settingsRMS.getRecord(1)!=null)
						result = new String(settingsRMS.getRecord(1));
				}
				//result = new String (settingsRMS.getRecord(1));
				settingsRMS.closeRecordStore();
			}
			catch (RecordStoreException e){
				System.out.println(e.toString());
				return EMPTY;
			}
			if (result == null) return EMPTY;
			return result; 
		}	
		
		public void initChangeSettings(Displayable back, String message){
			ret_screen=back;
			settingsForm = new Form ("Settings");
			
			// read old data
			String  credential = globalPreferences.get(EGOBAN_USER);
			if (credential != EMPTY)
				egobanUsernameField = new TextField("Egoban Usernamame:", credential, 42, TextField.ANY);
			else
				egobanUsernameField = new TextField("Egoban Usernamame:", "", 42, TextField.ANY);
			credential = globalPreferences.get(EGOBAN_PASSWD);
			if (credential != EMPTY)
				egobanPasswordField = new TextField("Egoban Password:", credential, 42, TextField.ANY);
			else
				egobanPasswordField = new TextField("Egoban Password:", "", 42, TextField.ANY);
			
			if (message != null) settingsForm.append(new StringItem("Hint:", message));
			
			startupHelpScreen = new ChoiceGroup("Show Help at Start?", Choice.EXCLUSIVE);
			startupHelpScreen.append("Yes",null);
			startupHelpScreen.append("No",null);

			int startupSelection = Integer.parseInt(globalPreferences.get(SHOW_HELP_AT_STARTUP));
			if (startupSelection>0)
				startupHelpScreen.setSelectedIndex(startupSelection, true);
			else
				startupHelpScreen.setSelectedIndex(0, true);
			
			settingsForm.append(egobanUsernameField);
			settingsForm.append(egobanPasswordField);
			settingsForm.append(startupHelpScreen);
			settingsForm.setCommandListener(this);
			settingsForm.addCommand(cmd_ok);
			settingsForm.addCommand(cmd_cancel);
			display.setCurrent(settingsForm);
		}
		
		public void commandAction (Command c, Displayable d)
		{
			if (c == cmd_ok) {
				globalPreferences.set(EGOBAN_USER, egobanUsernameField.getString());
				globalPreferences.set(EGOBAN_PASSWD, egobanPasswordField.getString());
				int startUpInt = startupHelpScreen.getSelectedIndex();
				globalPreferences.set(SHOW_HELP_AT_STARTUP,String.valueOf(startUpInt));
				display.setCurrent(ret_screen);
			}
			if (c == cmd_cancel){
				display.setCurrent(ret_screen);
			}
		}
		
		
	}

	public static class GoBrett {
		Stack sequence; // sequence of moves - pass = null
		Stack captured; // corresponding sequence of Vectors of captured stones (null for pass, empty Vector for no capture)
		int[][] board; // actual gamestate
		
		Coords koPosition = null;
		
		int color = BLACK; // current player's color
		int boardSize;
		int[] captives = {0 /* neutral */, 0 /* black */, 0 /* white */};
		
		// possible content of any given coordinate
		public static final int BLACK = 1;
		public static final int WHITE = 2;
		public static final int EMPTY = 0;
		public static final int OUTSIDE = -1;
		public static final int MARK_NEUTRAL = 3;
		public static final int MARK_BLACK = 4;
		public static final int MARK_WHITE = 5;
		
		// possible outcome of a move
		public static final int M_NORMAL = 1; // move is successful
		public static final int M_CAPTURE = 2; // unused, TODO remove it
		public static final int M_SUICIDE = 3; // move would result in suicide
		public static final int M_KO = 4; // move would violate Ko rule
		public static final int M_INVALID = 0; // move is invalid (e.g. goes outside the board or on top of a stone)
		
		public int enemy (int color) { return (color == BLACK) ? WHITE : BLACK; }
		public void flipColor () { color = enemy(color); } // remember to call this at the end of each turn
			// XXX shouldn't this be private?
		
		protected boolean review = false; // review mode on/off switch
		protected int reviewIndex = 0; // current position in review
		
		public void beginReview () // start the review mode
		{
			if (review) endReview ();
			review = true;
			reviewIndex = sequence.size();
		}
		
		// go back/forward one move.
		// returns true it step was successful,
		// false if review mode is off or there's nowhere to go
		public boolean back ()
		{
			if (!review) return false;
			if (reviewIndex <= 0) return false;
			reviewIndex--;
			Coords st = (Coords)sequence.elementAt(reviewIndex);
			Vector gr = (Vector)captured.elementAt(reviewIndex);
			if (st != null) {
				board[st.x][st.y] = EMPTY;
				putStones (gr, color);
			}
			flipColor();
			return true;
		}
		public boolean forward ()
		{
			if (!review) return false;
			if (reviewIndex >= sequence.size()) return false;
			Coords st = (Coords)sequence.elementAt(reviewIndex);
			Vector gr = (Vector)captured.elementAt(reviewIndex);
			if (st != null) {
				board[st.x][st.y] = color;
				removeStones(gr);
			}
			flipColor();
			reviewIndex++;
			return true;
		}
		
		// go forward/back i steps
		public void forward (int i) { while (i-- > 0 && forward()) ; }
		public void back (int i) { while (i-- > 0 && back()) ; }
		
		// return to play mode, restore board state
		public void endReview ()
		{
			while (forward()) ;
			review = false;
		}
		
		// coordinates of the last move (null for pass or for start of game)
		public Coords lastMove ()
		{
			if (review) return (reviewIndex>0)?(Coords)sequence.elementAt(reviewIndex-1):null;
			else if (sequence.empty()) return null;
			else return (Coords)sequence.peek();
		}
		
		public GoBrett (String sgf) {
			// create GoBrett from SGF string
			
			// converts main game (no variation) from SGF to Move-Position-HashTable
			
			color = BLACK;
			for (int i = 0; i < captives.length; i++) captives[i] = 0;
			sequence = new Stack(); captured = new Stack();
			
			/* Parse SGF to
			 1) Get boardsize
			 2) return moves from main line of game (no variations) 
			    which will later be used to fill GoBrett's values (sequence/captured etc.)			 
			 */
			
			// find boardsize
			int sz = sgf.indexOf("SZ[");
			if (sz > -1) {
				sz += 3;
				int end = sgf.indexOf(']',sz);
				boardSize = Integer.parseInt(sgf.substring(sz,end));
			} else {
				// assume 19
				boardSize = 19;
			}
			
			board = new int[boardSize][boardSize];
			
			//find moves
		
			// TODO find out how SGF stores "pass"
			for (int i=0; i<sgf.length()-1; i++){
				char aktChar=sgf.charAt(i);
					// if B/W detected: check for coordinates and save them in mySequence
					if ((aktChar=='B' ||  aktChar=='W') && 
							(sgf.charAt(i+1)=='[') 
							&& sgf.charAt(i+4)==']'){
						
						play (new Coords(sgf.charAt(i+2),sgf.charAt(i+3)));
					}
			}
		}
		
		public String toSGF ()
		{
			// return string containing SGF representation of this game

			// SGF intro
			StringBuffer sgf = new StringBuffer("(;FF[4]GM[1]AP[mgo_x "+cvs_Id+"]SZ["+boardSize+"]PW[White]PB[Black]");
			
			char[] colors = { 0, 'B', 'W' };
			int color = BLACK;
			
			for (int i = 0; i < sequence.size(); i++) {
				Coords c = (Coords)sequence.elementAt(i);
				if (c != null) { 
					sgf.append(';');
					sgf.append(colors[color]);
					sgf.append('['); sgf.append(c.toSGF()); sgf.append(']');
				}
				color = enemy(color);
			}
			sgf.append(')');
			
			return sgf.toString();
		}

		public GoBrett (int boardSize)
		{
			this.boardSize = boardSize;
			board = new int[boardSize][boardSize];
			sequence = new Stack();
			captured = new Stack();
		}
		
		public GoBrett (GoBrett orig)
		{
			sequence = new Stack(); sequence.ensureCapacity(orig.sequence.size());
			for (int i = 0; i < orig.sequence.size(); i++) sequence.push(orig.sequence.elementAt(i));
			captured = new Stack(); captured.ensureCapacity(orig.captured.size());
			for (int i = 0; i < orig.captured.size(); i++) captured.push(orig.captured.elementAt(i));
			boardSize = orig.boardSize;
			color = orig.color;
			board = new int[boardSize][boardSize];
			for (int i = 0; i < boardSize; i++) for (int j = 0; j < boardSize; j++)
				board[i][j] = orig.board[i][j];
			for (int i = 0; i < captives.length; i++) captives[i] = orig.captives[i];
			koPosition = orig.koPosition;
		}
		
		public GoBrett (RecordStore saveRecordStore)
		{
			load(saveRecordStore);
		}
		
		public void load (RecordStore store)
		//throws RecordStoreException
		{
			// get number of records
			int recNum;
			try{
				recNum = store.getNumRecords();
			} catch (RecordStoreException e){
				throw new RuntimeException("getNumRecords failed: "+e.toString());
			}	
			
			if (recNum > 0) {
				byte[] b;
				
				// get boardsize
				try {
					b = store.getRecord(1);
					boardSize = b[0];
				} catch (RecordStoreException e) {
					throw new RuntimeException("getRecord(1) failed: "+e.toString());
				}
				board = new int[boardSize][boardSize];
				color = BLACK;
				for (int i = 0; i<captives.length; i++) captives[i] = 0;
				sequence = new Stack(); captured = new Stack();
				
				// replay the moves sequence
				// TODO: might be better to store the actual stacks and not need to replay
				for (int i = 2; i <= recNum; i++) try {
					b = store.getRecord(i);
					if (b==null) pass();
					else {
						Coords c = new Coords(b[0],b[1]);
						play (c);
					} 
				
				} catch (RecordStoreException e) {
					throw new RuntimeException("getRecord($i) failed: "+e.toString());
				}
			}
		}
		
		public void save (RecordStore store)
		throws RecordStoreException {
			//	store boardsize in RS
			byte[] b = new byte[2];
			
			b[0] = (byte)boardSize;
			store.addRecord(b,0,1);
			for (int i = 0; i < sequence.size(); i++) {
				Coords c = (Coords)sequence.elementAt(i);
				if (c == null) {
					store.addRecord(null,0,0);
				} else {
					b[0] = c.bx(); b[1] = c.by();
					store.addRecord(b,0,2);
				}
			}
		}
		
		// are the coordinates outside?
		public boolean outside (Coords c) { return outside(c.x,c.y); }
		public boolean outside (int x, int y) {
			int min = x>y ? y : x; int max = x>y ? x : y;
			if (min < 0 || max >= boardSize) return true;
			return false;
		}

		// is there a stone on the coordinates?
		public boolean contains (Coords c) { return contains(c.x, c.y); }
		public boolean contains (int x, int y) {
			if (outside(x,y)) return false;
			return board[x][y] != EMPTY;
		}

		// get the position
		public int get (Coords c) { return get(c.x,c.y); }
		public int get (int x, int y) {
			if (outside(x,y)) return OUTSIDE;
			return board[x][y];
		}
		
		// is the position empty (i.e. possible liberty)?
		public boolean empty (Coords c) { return empty(c.x,c.y); }
		public boolean empty (int x, int y) {
			if (outside(x,y)) return false;
			return board[x][y] == EMPTY;
		}
		
		// how many free points are next to this position?
		public int individualLiberties(Coords pos) {	
			int ret = 0;
			if (empty(pos.x-1,pos.y)) ret++;
			if (empty(pos.x+1,pos.y)) ret++; 
			if (empty(pos.x,pos.y-1)) ret++;
			if (empty(pos.x,pos.y+1)) ret++;
			return ret;
		}

		public boolean groupKilled (Coords pos) {
			// returns true if the selected group has no liberties, false otherwise
			if (!contains(pos)) return false;
			int color = board[pos.x][pos.y];
			Hashtable group = new Hashtable();
			Stack unchecked = new Stack();

			unchecked.push(pos);
			Integer is = new Integer(0);

			// for explanation see getGroup
			while (!unchecked.isEmpty()) {
				Coords actual = (Coords)unchecked.pop();
				if (individualLiberties(actual)>0) return false;
				
				group.put(actual.toSGF(),is);
				Stack touch = touching (actual,color);
				while (!touch.isEmpty()) {
					Coords stone = (Coords)touch.pop();
					if (!group.containsKey(stone.toSGF())) unchecked.push(stone);
				}
			}
			return true;
		}
		
		private Vector getGroup (Coords pos) {
			// returns coordinates of all stones belonging to a group of same value (even empty or marked)
			//System.out.println("getGroup of position (x/y) "+pos.x+"/"+pos.y);
			if (outside(pos)) return null;
			int color = get(pos);
			Vector ret = new Vector(); // the result vector
			Hashtable group = new Hashtable(); // ids of stones already known
			Stack unchecked = new Stack(); // stack of stones to be checked

			unchecked.push (pos); // start with our coordinate
			Integer is = new Integer(0); // fake value to insert into hashtable

			while (!unchecked.isEmpty()) {
				// test each unchecked:
				Coords actual = (Coords)unchecked.pop();
				if (!group.containsKey(actual.toSGF())) { // not sure whether this should contain the while-loop below as well
					group.put(actual.toSGF(),is); // save its ident to the hashtable (note that Coords don't have hash method, String does)
				
					ret.addElement(actual);
				}

				// find neighbours of the same color
				Stack touch = touching(actual,color);
				while (!touch.isEmpty()) {
					Coords stone = (Coords)touch.pop();
					//System.out.println("stone.x/y"+stone.x+"/"+stone.y);
					if (!group.containsKey(stone.toSGF())) // if this neighbour is unknown (we didn't hit it before)
						unchecked.push(stone); // add it to be checked
				
						
				}
			}
			return ret;
		}
		
		// paint a group of points by a chosen color
		public /* XXX to be protected */ void removeStones (Vector group) { putStones(group, EMPTY); }
		public /* XXX to be protected */ void putStones (Vector group, int color)
		{
			for (int i = 0; i < group.size(); i++) {
				Coords stone = (Coords)group.elementAt(i);
				if (outside(stone)) continue;
				board[stone.x][stone.y] = color;
			}
		}
		
		// remove group from board, count captives
		public void removeGroup (Coords pos) {
			int color = get(pos);
			Vector group = getGroup(pos);
			removeStones(group);
			if (color < captives.length) captives[enemy(color)] += group.size();
		}
		
		// find neighbours of 'pos' with the given value 'color'
		public Stack touching(Coords pos, int color) {
			Stack ret = new Stack ();
			Coords c;
			if (get(c = new Coords(pos.x-1,pos.y))==color)
				ret.push(c);
			if (get(c = new Coords(pos.x+1,pos.y))==color)
				ret.push(c);
			if (get(c = new Coords(pos.x,pos.y-1))==color)
				ret.push(c);
			if (get(c = new Coords(pos.x,pos.y+1))==color)
				ret.push(c);
			return ret;
		}
		
		// how many color'ed neighbours does pos have?
		public int countTouching (Coords pos, int color) {
			int ret = 0;
			if (get(pos.x-1,pos.y)==color) ret++;
			if (get(pos.x+1,pos.y)==color) ret++;
			if (get(pos.x,pos.y-1)==color) ret++;
			if (get(pos.x,pos.y+1)==color) ret++;
			return ret;
		}

		// pass a move
		public void pass() {
			if (review) return;
			sequence.push(null);
			captured.push(null);
			koPosition = null;
			flipColor();
		}

		// play on pos_
		public int play (Coords pos_) {
			Coords pos = new Coords(pos_); // make a clone - remember that DrawingGoban always plays with the same Coords object
			if (review) return M_INVALID;
			if (!empty(pos)) return M_INVALID;

			if (pos.equals(koPosition)) return M_KO;
			
			// try the move
			board[pos.x][pos.y] = color;
			
			// capture - are any enemy groups dead now?
			Vector capts = new Vector();
			Stack others = touching(pos, enemy(color));
			while (!others.empty()) {
				Coords stone = (Coords)others.pop();
				if (groupKilled(stone)) {
					Vector group = getGroup (stone);
					removeStones (group);
					
					capts.ensureCapacity(capts.size()+group.size());
					for (int i = 0; i < group.size(); i++)
						capts.addElement(group.elementAt(i));
				}
			}
			
			// suicide - is a new group dead?
			if (groupKilled(pos)) {
				board[pos.x][pos.y] = EMPTY;
				putStones(capts, enemy(color)); // remember to replace the captives
					// not that this makes any sense - if there were captives, then at least one liberty was freed
					// ...but just in case
				return M_SUICIDE;
			}
			
			// the move is ok now.

			// count captives, add to sequence
			captives[color] += capts.size();
			//System.out.println("capts.size()="+capts.size());
			sequence.push(pos);
			captured.push(capts);
			
			// ko check:
			koPosition = null;
			if (capts.size()==1) { // we took only one stone, enemy can replace it
				Coords ko = (Coords)capts.elementAt(0);
				if ( countTouching(pos,color) == 0 // no stones of our color touch the position
					&& individualLiberties(pos) == 1) { // and the stone has only one liberty
					/* now the situation is as follows:
					 a) we placed a stone that's not connected to anything
					 b) that stone captured exactly one enemy stone, thus gaining one liberty
					 c) now it has only one liberty, so it couldn't have had any liberties before.
					    Moreover, it can be captured by placing a single stone on that liberty.
					 d) if that happened, it would remove only our new stone (it's not connected)
					    and replace the old captured one, while NOTHING ELSE would change.
					That can mean only one thing - a Ko!
					*/
					
					koPosition = ko;
				}
			}
			
			flipColor();
			pos = null;
			capts = null;
			return M_NORMAL;
		}
		
		// undo last move
		public void undo ()
		{
			if (review) return;
			if (sequence.isEmpty()) return;
			Vector removed = (Vector)captured.pop();
			Coords stone = (Coords)sequence.pop();
			
			if (removed != null) {
				captives[enemy(color)] -= removed.size();
				putStones(removed, color);
			}
			if (stone != null)
				board[stone.x][stone.y] = EMPTY;
			
			// ko check, see play() for explanation
			koPosition = null;
			if (!sequence.isEmpty()) {
				Coords prevSt = (Coords)sequence.peek();
				Vector prevG = (Vector)captured.peek();
				if (prevSt != null && prevG != null &&
						prevG.size()==1 && countTouching(prevSt,get(prevSt))==0 && individualLiberties(prevSt)==1)
					koPosition = (Coords)prevG.elementAt(0);
			}
			
			flipColor();
		}

	} // end fo class GoBrett

		
	// Lifecycle methods:
	public void startApp() {
		Alert splash = new Alert("Welcome to mgo"); 
		
		if ( ! (globalPreferences.get(SHOW_HELP_AT_STARTUP).compareTo(Preferences.EMPTY) == 0))
			display.setCurrent(logoScreen);
		else {
			splash = new Alert("mgo - The Game of Go on Mobile Devices");
			splash.setString(		
					"Welcome to mgo!\n\n"+ 
					"Do you write protocols of your (offline) games for review and study?\n" +
					"Use mgo as a digital kifu sheet and " +
					"store your games digitally to take them with you wherever you go. "+
					"You can even "+
					"export them to your device's file system (using Go software compatible SGF files) " +
					"or " +
					"directly transfer "+
					"them over the air to \"egoban.org\" (a free colaborative online Go service).\n\n" +
					"Do you fancy playing a quick two-player match "+
					"with your friends?\n" +
					"With mgo you can give Go a go - wherever you are.\n\n" +
					"For most recent releases and other information visit\n www.mobilego.org\n\n"+
					"You can disable this information in the mgo settings.\n\n"+
					"(Press Done/Ok to continue)");
			splash.setTimeout(splash.FOREVER);
			display.setCurrent(splash, mainMenu);
		}
	}

	protected void pauseApp() {
	}

	protected void destroyApp(boolean unconditional) {
	}

}
